<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\JobPosting;
use Illuminate\Http\Request;

class BookingController extends Controller
{
    // Worker applies for a job
    public function apply(Request $request, JobPosting $job)
    {
        if ($request->user()->role !== 'worker') {
            return response()->json(['message' => 'Only workers can apply'], 403);
        }

        if ($request->user()->status !== 'active') {
            return response()->json(['message' => 'Your account must be verified to apply for jobs'], 403);
        }
        
        // Prevent double application
        if (Booking::where('job_id', $job->id)->where('worker_id', $request->user()->id)->exists()) {
            return response()->json(['message' => 'Already applied'], 400);
        }

        $booking = Booking::create([
            'job_id' => $job->id,
            'worker_id' => $request->user()->id,
            'status' => 'applied',
        ]);

        return response()->json($booking, 201);
    }

    // Client accepts a worker
    public function accept(Request $request, Booking $booking)
    {
        // Verify client owns the job
        if ($booking->job->client_id !== $request->user()->id) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $booking->update(['status' => 'approved']);
        $booking->job->update(['status' => 'filled']); // Simple logic: 1 job = 1 worker for now

        return response()->json(['message' => 'Worker accepted']);
    }

    // Worker submits timesheet
    public function submitTimesheet(Request $request, Booking $booking)
    {
        if ($booking->worker_id !== $request->user()->id) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
            'start_time' => 'required|date',
            'end_time' => 'required|date|after:start_time',
            'break_minutes' => 'nullable|integer',
        ]);

        $booking->update([
            'actual_start_time' => $request->start_time,
            'actual_end_time' => $request->end_time,
            'break_minutes' => $request->break_minutes ?? 0,
            'status' => 'completed_pending_review',
        ]);

        return response()->json(['message' => 'Timesheet submitted']);
    }

    // Client approves timesheet
    public function approveTimesheet(Request $request, Booking $booking)
    {
         if ($booking->job->client_id !== $request->user()->id) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $booking->update(['status' => 'completed_approved']);
        // Trigger payment calculation logic here later

        return response()->json(['message' => 'Timesheet approved']);
    }
    
    // Get bookings for current user
    public function index(Request $request)
    {
        $user = $request->user();
        if ($user->role === 'worker') {
            return Booking::with(['job.client.clientProfile'])->where('worker_id', $user->id)->latest()->get();
        } else {
             return Booking::with(['job', 'worker', 'worker.workerProfile'])->whereHas('job', function($q) use ($user) {
                 $q->where('client_id', $user->id);
             })->latest()->get();
        }
    }
}
