import { defineStore } from 'pinia';
import axios from 'axios';
import router from '../router';

export const useAuthStore = defineStore('auth', {
    state: () => ({
        user: null,
        token: localStorage.getItem('token') || null,
        loading: false,
        error: null,
    }),

    getters: {
        isAuthenticated: (state) => !!state.token,
        isWorker: (state) => state.user?.role === 'worker',
        isClient: (state) => state.user?.role === 'client',
        isAdmin: (state) => state.user?.role === 'admin',
    },

    actions: {
        async register(userData) {
            this.loading = true;
            this.error = null;
            try {
                const response = await axios.post('/api/register', userData);
                this.token = response.data.access_token;
                this.user = response.data.user;
                localStorage.setItem('token', this.token);
                axios.defaults.headers.common['Authorization'] = `Bearer ${this.token}`;

                this.redirectUser();
            } catch (error) {
                this.error = error.response?.data?.message || 'Registration failed';
                throw error;
            } finally {
                this.loading = false;
            }
        },

        async login(credentials) {
            this.loading = true;
            this.error = null;
            try {
                const response = await axios.post('/api/login', credentials);
                this.token = response.data.access_token;
                this.user = response.data.user;
                localStorage.setItem('token', this.token);
                axios.defaults.headers.common['Authorization'] = `Bearer ${this.token}`;

                await this.fetchUser(); // Get full profile
                this.redirectUser();
            } catch (error) {
                this.error = error.response?.data?.message || 'Login failed';
                throw error;
            } finally {
                this.loading = false;
            }
        },

        async fetchUser() {
            if (!this.token) return;
            try {
                const response = await axios.get('/api/profile');
                this.user = response.data;
            } catch (error) {
                this.logout();
            }
        },

        async logout() {
            try {
                await axios.post('/api/logout');
            } catch (e) {
                // Ignore error on logout
            } finally {
                this.user = null;
                this.token = null;
                localStorage.removeItem('token');
                delete axios.defaults.headers.common['Authorization'];
                // Force full reload to clear all state
                window.location.href = '/login';
            }
        },

        async updateWorkerProfile(data) {
            try {
                const response = await axios.post('/api/profile/worker', data);
                this.user = response.data.user; // Update local user state
                return response.data;
            } catch (error) {
                throw error;
            }
        },

        async updateClientProfile(data) {
            try {
                const response = await axios.post('/api/profile/client', data);
                this.user = response.data.user; // Update local user state
                return response.data;
            } catch (error) {
                throw error;
            }
        },

        async forgotPassword(email) {
            this.loading = true;
            this.error = null;
            try {
                const response = await axios.post('/api/forgot-password', { email });
                return response.data;
            } catch (error) {
                this.error = error.response?.data?.message || 'Failed to send reset link';
                throw error;
            } finally {
                this.loading = false;
            }
        },

        async resetPassword(payload) {
            this.loading = true;
            this.error = null;
            try {
                const response = await axios.post('/api/reset-password', payload);
                return response.data;
            } catch (error) {
                this.error = error.response?.data?.message || 'Failed to reset password';
                throw error;
            } finally {
                this.loading = false;
            }
        },

        redirectUser() {
            if (this.isWorker) router.push({ name: 'worker.dashboard' });
            else if (this.isClient) router.push({ name: 'client.dashboard' });
            else if (this.isAdmin) router.push({ name: 'admin.dashboard' });
            else router.push({ name: 'home' });
        }
    }
});
